import {
  Button,
  TextField,
  Badge,
  PasswordInput,
  Spinner,
  Checkbox,
  useConfirmation,
  useToast,
} from '@idds/react';
import { IconDownload, IconCheck, IconArrowRight } from '@tabler/icons-react';
import { useState } from 'react';

function App() {
  const [name, setName] = useState('');
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [termsAccepted, setTermsAccepted] = useState(false);
  const { confirm } = useConfirmation();
  const { toast } = useToast();

  const handleNameChange = (val: string) => {
    setName(val);
  };

  const handleEmailChange = (val: string) => {
    setEmail(val);
  };

  const handlePasswordChange = (val: string) => {
    setPassword(val);
  };

  const handleSubmit = async () => {
    const ok = await confirm({
      title: 'Konfirmasi',
      message: 'Apakah Anda yakin ingin mengirim data ini?',
      confirmText: 'Kirim',
      cancelText: 'Batal',
    });

    if (ok) {
      setIsSubmitting(true);
      // Simulate API call
      await new Promise((resolve) => setTimeout(resolve, 1000));
      setIsSubmitting(false);
      toast({
        title: 'Data berhasil dikirim!',
        state: 'positive',
        position: 'top-right',
        duration: 3000,
      });
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 p-8">
      <div className="max-w-4xl mx-auto space-y-8">
        {/* Header */}
        <div className="text-center space-y-4">
          <h1 className="text-4xl font-bold text-content-primary">
            INA Digital Design System
          </h1>
          <p className="text-lg text-content-secondary">
            React Starter Project dengan Tailwind CSS v4
          </p>
        </div>

        {/* Card Example */}
        <div className="space-y-4">
          <p className="text-sm text-content-secondary">
            Project ini sudah dikonfigurasi dengan:
          </p>
          <div className="flex flex-wrap gap-2">
            <Badge variant="info">React 18</Badge>
            <Badge variant="warning">Vite</Badge>
            <Badge variant="success">Tailwind CSS v4</Badge>
            <Badge variant="error">TypeScript</Badge>
          </div>
        </div>

        {/* Button Examples */}
        <div className="space-y-4">
          <div className="flex flex-wrap gap-3">
            <Button hierarchy="primary" size="md">
              Primary Button
            </Button>
            <Button hierarchy="secondary" size="md">
              Secondary Button
            </Button>
            <Button hierarchy="tertiary" size="md">
              Tertiary Button
            </Button>
          </div>

          <div className="flex flex-wrap gap-3">
            <Button hierarchy="primary" prefixIcon={<IconDownload size={16} />}>
              Download
            </Button>
            <Button
              hierarchy="secondary"
              suffixIcon={<IconArrowRight size={16} />}
            >
              Continue
            </Button>
            <Button
              hierarchy="primary"
              prefixIcon={<IconCheck size={16} />}
              suffixIcon={<IconArrowRight size={16} />}
            >
              Submit
            </Button>
          </div>

          <div className="flex flex-wrap gap-3">
            <Button hierarchy="primary" disabled>
              Disabled
            </Button>
            <Button hierarchy="secondary" disabled>
              Disabled
            </Button>
          </div>
        </div>

        {/* TextField Example */}
        <div className="space-y-4 w-full sm:max-w-3/4 md:max-w-2/3 lg:max-w-1/2">
          <TextField
            label="Nama Lengkap"
            placeholder="Masukkan nama lengkap Anda"
            size="md"
            value={name}
            onChange={handleNameChange}
          />
          <TextField
            label="Email"
            type="email"
            placeholder="nama@example.com"
            size="md"
            value={email}
            onChange={handleEmailChange}
          />
          <PasswordInput
            label="Password"
            placeholder="Masukkan password"
            size="md"
            value={password}
            onChange={handlePasswordChange}
          />

          <Checkbox
            id="terms-checkbox"
            label="Saya menyetujui syarat dan ketentuan"
            checked={termsAccepted}
            onChange={(checked) => setTermsAccepted(checked)}
          />

          <Button
            hierarchy="primary"
            size="md"
            onClick={handleSubmit}
            disabled={isSubmitting || !termsAccepted}
          >
            {isSubmitting ? (
              <>
                <span style={{ marginRight: 8, display: 'inline-flex' }}>
                  <Spinner size={16} spinnerOnly />
                </span>
                Mengirim...
              </>
            ) : (
              <>
                <IconCheck size={16} style={{ marginRight: 8 }} />
                Submit
                <IconArrowRight size={16} style={{ marginLeft: 8 }} />
              </>
            )}
          </Button>
        </div>

        {/* Footer */}
        <div className="text-center text-sm text-content-tertiary">
          <p className="text-content-secondary">
            Dokumentasi lengkap:{' '}
            <a
              href="https://design.inadigital.co.id"
              className="text-guide-500 hover:text-guide-600 underline"
              target="_blank"
              rel="noopener noreferrer"
            >
              design.inadigital.co.id
            </a>
          </p>
        </div>
      </div>
    </div>
  );
}

export default App;
